<?php
namespace App;

class Cache
{
    private static function path(string $key): string
    {
        $safe = preg_replace('/[^a-zA-Z0-9_\-]/', '_', $key);
        $dir = __DIR__ . '/../storage/cache';
        if (!is_dir($dir)) @mkdir($dir, 0777, true);
        return $dir . '/' . $safe . '.cache.php';
    }

    public static function get(string $key)
    {
        $path = self::path($key);
        if (!is_file($path)) return null;
        $data = include $path;
        if (!is_array($data)) return null;
        if (!empty($data['expires_at']) && $data['expires_at'] < time()) {
            @unlink($path);
            return null;
        }
        return $data['value'] ?? null;
    }

    public static function set(string $key, $value, int $ttlSec = 300): void
    {
        $path = self::path($key);
        $payload = var_export([
            'expires_at' => time() + $ttlSec,
            'value' => $value,
        ], true);
        file_put_contents($path, "<?php\nreturn {$payload};\n");
    }

    public static function remember(string $key, int $ttlSec, callable $callback)
    {
        $existing = self::get($key);
        if ($existing !== null) return $existing;
        $value = $callback();
        self::set($key, $value, $ttlSec);
        return $value;
    }
}

