<?php
namespace App\Controllers;

use App\View;
use App\Database;
use App\Auth;

class AuthController
{
    public function loginForm(): void
    {
        echo View::render('auth/login', [
            'title' => 'Login — BlocFolio',
            'csrf' => Auth::csrfToken(),
        ]);
    }

    public function loginPost(): void
    {
        $email = strtolower(trim($_POST['email'] ?? ''));
        $pass = $_POST['password'] ?? '';
        $csrf = $_POST['csrf'] ?? '';
        if (!Auth::checkCsrf($csrf)) { http_response_code(400); echo 'Bad CSRF'; return; }
        // Throttle: 5 attempts per 15 minutes per email+IP
        $ip = $_SERVER['REMOTE_ADDR'] ?? '';
        $pdo = Database::pdo();
        $window = new \DateTime('-15 minutes');
        $stmt = $pdo->prepare('SELECT id, attempts, window_start FROM login_attempts WHERE email = :e AND ip = :ip LIMIT 1');
        $stmt->execute([':e'=>$email, ':ip'=>$ip]);
        $row = $stmt->fetch();
        if ($row) {
            $attempts = (int)$row['attempts'];
            $win = new \DateTime($row['window_start']);
            if ($win < $window) { $attempts = 0; }
            if ($attempts >= 5) { http_response_code(429); echo 'Too many attempts. Try later.'; return; }
        }
        // Optional reCAPTCHA
        $recaptchaEnabled = (bool)\App\Config::get('recaptcha.enabled', false);
        if ($recaptchaEnabled) {
            $token = $_POST['g-recaptcha-response'] ?? '';
            $secret = \App\Config::get('recaptcha.secret');
            if (!$token || !$secret) { http_response_code(400); echo 'Captcha required'; return; }
            // Verify via Google
            $resp = @file_get_contents('https://www.google.com/recaptcha/api/siteverify?secret=' . urlencode($secret) . '&response=' . urlencode($token) . '&remoteip=' . urlencode($ip));
            $ok = false;
            if ($resp) { $data = json_decode($resp, true); $ok = !empty($data['success']); }
            if (!$ok) { http_response_code(400); echo 'Captcha failed'; return; }
        }
        $stmt = $pdo->prepare('SELECT id, password_hash FROM users WHERE email = :e LIMIT 1');
        $stmt->execute([':e'=>$email]);
        $u = $stmt->fetch();
        if (!$u || empty($u['password_hash']) || !password_verify($pass, $u['password_hash'])) {
            // bump attempts
            if ($row) {
                $pdo->prepare('UPDATE login_attempts SET attempts = attempts + 1, window_start = NOW() WHERE id = :id')->execute([':id'=>(int)$row['id']]);
            } else {
                $pdo->prepare('INSERT INTO login_attempts (email, ip, attempts, window_start) VALUES (:e,:ip,1,NOW())')->execute([':e'=>$email, ':ip'=>$ip]);
            }
            echo View::render('auth/login', ['title'=>'Login — BlocFolio','error'=>'Invalid credentials','csrf'=>Auth::csrfToken()]);
            return;
        }
        // success: reset attempts
        if ($row) { $pdo->prepare('UPDATE login_attempts SET attempts = 0, window_start = NOW() WHERE id = :id')->execute([':id'=>(int)$row['id']]); }
        Auth::login((int)$u['id']);
        header('Location: /');
    }

    public function registerForm(): void
    {
        echo View::render('auth/register', [
            'title' => 'Register — BlocFolio',
            'csrf' => Auth::csrfToken(),
        ]);
    }

    public function registerPost(): void
    {
        $email = strtolower(trim($_POST['email'] ?? ''));
        $name = trim($_POST['name'] ?? '');
        $pass = $_POST['password'] ?? '';
        $csrf = $_POST['csrf'] ?? '';
        if (!Auth::checkCsrf($csrf)) { http_response_code(400); echo 'Bad CSRF'; return; }
        if (!filter_var($email, FILTER_VALIDATE_EMAIL) || strlen($pass) < 6) {
            echo View::render('auth/register', ['title'=>'Register — BlocFolio','error'=>'Invalid input','csrf'=>Auth::csrfToken()]);
            return;
        }
        $pdo = Database::pdo();
        try {
            $pdo->prepare('INSERT INTO users (email, password_hash, name) VALUES (:e,:p,:n)')
                ->execute([':e'=>$email, ':p'=>password_hash($pass, PASSWORD_BCRYPT), ':n'=>$name]);
        } catch (\Throwable $e) {
            echo View::render('auth/register', ['title'=>'Register — BlocFolio','error'=>'Email already used','csrf'=>Auth::csrfToken()]);
            return;
        }
        $id = (int)$pdo->lastInsertId();
        Auth::login($id);
        header('Location: /');
    }

    public function logout(): void
    {
        Auth::logout();
        header('Location: /');
    }
}
