<?php
namespace App\Controllers;

use App\View;
use App\Models\Coin;
use App\Config;
use App\SEO;
use App\Snapshot;

class CoinsController
{
    public function index(): void
    {
        $page = max(1, (int)($_GET['page'] ?? 1));
        $limit = min(100, max(10, (int)($_GET['limit'] ?? 25)));
        $sort = $_GET['sort'] ?? 'rank';
        $dir = strtolower($_GET['dir'] ?? 'asc') === 'desc' ? 'desc' : 'asc';
        $filters = [
            'q' => $_GET['q'] ?? null,
            'category' => $_GET['category'] ?? null,
        ];
        [$rows, $total] = Coin::paginated($page, $limit, $sort, $dir, $filters);
        $meta = \App\SEO::listCoinsMeta();
        $html = View::render('coins/list', [
            'title' => $meta['title'] ?? 'Coins — BlocFolio',
            'meta' => $meta,
            'coins' => $rows,
            'total' => $total,
            'page' => $page,
            'limit' => $limit,
            'sort' => $sort,
            'dir' => $dir,
        ]);
        if (\App\Config::get('app.env') === 'prod' && (\App\Config::get('seo.snapshots.enable_lists', true))) {
            $ttl = (int)\App\Config::get('seo.snapshots.ttl.coins', 10800);
            \App\Snapshot::write('/coins', $html, $ttl);
        }
        echo $html;
    }

    public function show(string $slug): void
    {
        $coin = Coin::findBySlug($slug);
        if (!$coin) {
            http_response_code(404);
            echo View::render('errors/404', ['title' => 'Coin Not Found']);
            return;
        }
        $meta = SEO::coinMeta($coin, Config::get('app.currency_default', 'USD'));
        $html = View::render('coins/show', [
            'title' => $meta['title'] ?? (($coin['name'] ?? 'Coin') . ' — BlocFolio'),
            'meta' => $meta,
            'coin' => $coin,
            'currency' => Config::get('app.currency_default', 'USD')
        ]);
        if (Config::get('app.env') === 'prod') {
            $route = '/coins/' . ($coin['slug'] ?? '');
            Snapshot::write($route, $html, 6*3600);
        }
        echo $html;
    }
}
