<?php
namespace App\Controllers;

use App\Auth;
use App\Config;
use App\Database;
use App\ScreenEngine;
use App\Utils;
use App\View;

class ScreensController
{
    public function index(): void
    {
        $pdo = Database::pdo();
        $user = Auth::user();
        $sort = $_GET['sort'] ?? 'recent';
        if ($sort === 'stars') {
            $pub = $pdo->query("SELECT s.id, s.name, s.slug, s.visibility, COUNT(ss.user_id) AS stars
                                FROM screens s LEFT JOIN screen_stars ss ON ss.screen_id = s.id
                                WHERE s.visibility = 'public'
                                GROUP BY s.id ORDER BY stars DESC, s.id DESC LIMIT 100")->fetchAll();
        } else {
            $pub = $pdo->query("SELECT s.id, s.name, s.slug, s.visibility, (SELECT COUNT(*) FROM screen_stars ss WHERE ss.screen_id = s.id) AS stars
                                FROM screens s WHERE s.visibility = 'public' ORDER BY s.id DESC LIMIT 100")->fetchAll();
        }
        $mine = [];
        if ($user) {
            $stmt = $pdo->prepare("SELECT id, name, slug, visibility FROM screens WHERE user_id = :u ORDER BY id DESC LIMIT 50");
            $stmt->execute([':u'=>(int)$user['id']]);
            $mine = $stmt->fetchAll();
        }
        echo View::render('screens/index', [
            'title' => 'Screens — BlocFolio',
            'public' => $pub,
            'mine' => $mine,
        ]);
    }

    public function createForm(): void
    {
        $user = Auth::requireUser();
        echo View::render('screens/new', [
            'title' => 'New Screen — BlocFolio',
            'csrf' => Auth::csrfToken(),
            'sample' => "mcap > 1000000000 AND change_24h > 5 ORDER BY change_7d DESC LIMIT 50",
        ]);
    }

    public function create(): void
    {
        $user = Auth::requireUser();
        if (!Auth::checkCsrf($_POST['csrf'] ?? '')) { http_response_code(400); echo 'Bad CSRF'; return; }
        $name = trim($_POST['name'] ?? '');
        $visibility = $_POST['visibility'] ?? 'private';
        $dsl = trim($_POST['dsl'] ?? '');
        if ($name === '' || $dsl === '') { header('Location: /screens/new'); return; }
        $slug = Utils::slug($name);
        $pdo = Database::pdo();
        // ensure unique slug
        $i = 1; $base = $slug;
        while ((int)$pdo->query("SELECT COUNT(*) FROM screens WHERE slug = " . $pdo->quote($slug))->fetchColumn() > 0) {
            $i++; $slug = $base . '-' . $i;
        }
        $def = ['dsl'=>$dsl];
        $stmt = $pdo->prepare("INSERT INTO screens (user_id, name, slug, visibility, definition_json, sort, `limit`, currency) VALUES (:u,:n,:s,:v,:j,NULL,NULL,:cur)");
        $stmt->execute([
            ':u'=>(int)$user['id'], ':n'=>$name, ':s'=>$slug, ':v'=>$visibility,
            ':j'=>json_encode($def), ':cur'=>Config::get('app.currency_default','USD')
        ]);
        header('Location: /screens/' . $slug);
    }

    public function show(string $slug): void
    {
        $pdo = Database::pdo();
        $stmt = $pdo->prepare('SELECT * FROM screens WHERE slug = :s LIMIT 1');
        $stmt->execute([':s'=>$slug]);
        $screen = $stmt->fetch();
        if (!$screen) { http_response_code(404); echo View::render('errors/404', ['title'=>'Screen Not Found']); return; }
        $def = json_decode($screen['definition_json'] ?? '{}', true) ?: [];
        $dsl = $def['dsl'] ?? '';
        $currency = $screen['currency'] ?: Config::get('app.currency_default','USD');
        $rows = [];
        if ($dsl) { $rows = ScreenEngine::run($dsl, $currency, 200); }
        echo View::render('screens/show', [
            'title' => $screen['name'] . ' — BlocFolio',
            'screen' => $screen,
            'rows' => $rows,
        ]);
    }

    public function exportCsv(string $slug): void
    {
        $pdo = Database::pdo();
        $stmt = $pdo->prepare('SELECT * FROM screens WHERE slug = :s LIMIT 1');
        $stmt->execute([':s'=>$slug]);
        $screen = $stmt->fetch();
        if (!$screen) { http_response_code(404); echo 'Not found'; return; }
        $def = json_decode($screen['definition_json'] ?? '{}', true) ?: [];
        $dsl = $def['dsl'] ?? '';
        $currency = $screen['currency'] ?: Config::get('app.currency_default','USD');
        $rows = $dsl ? ScreenEngine::run($dsl, $currency, 1000) : [];
        header('Content-Type: text/csv');
        header('Content-Disposition: attachment; filename="screen-'.preg_replace('/[^a-z0-9\-]/i','-', $screen['slug']).'.csv"');
        $out = fopen('php://output','w');
        fputcsv($out, ['rank','name','symbol','slug','price','market_cap','volume_24h','change_24h']);
        foreach ($rows as $r) {
            fputcsv($out, [
                $r['rank'] ?? '', $r['name'] ?? '', strtoupper($r['symbol'] ?? ''), $r['slug'] ?? '',
                $r['price'] ?? '', $r['market_cap'] ?? '', $r['volume_24h'] ?? '', $r['change_24h'] ?? ''
            ]);
        }
        fclose($out);
    }

    // AJAX preview
    public function preview(): void
    {
        header('Content-Type: application/json');
        $user = Auth::user(); // optional
        $dsl = trim($_GET['dsl'] ?? '');
        $currency = $_GET['currency'] ?? Config::get('app.currency_default','USD');
        if ($dsl === '') { echo json_encode(['data'=>[]]); return; }
        try {
            $rows = ScreenEngine::run($dsl, $currency, 100);
            echo json_encode(['data'=>$rows]);
        } catch (\Throwable $e) {
            http_response_code(400);
            echo json_encode(['error'=>'Invalid query']);
        }
    }
}
