<?php
namespace App\Controllers;

use App\Auth;
use App\Database;
use App\Utils;
use App\View;

class WatchlistsController
{
    public function index(): void
    {
        $user = Auth::requireUser();
        $pdo = Database::pdo();
        $rows = $pdo->prepare('SELECT id, name, slug, visibility, is_default FROM watchlists WHERE user_id = :u ORDER BY is_default DESC, id DESC');
        $rows->execute([':u'=>(int)$user['id']]);
        echo View::render('watchlists/index', [
            'title'=>'My Watchlists — BlocFolio', 'lists'=>$rows->fetchAll(), 'csrf'=>Auth::csrfToken()
        ]);
    }

    public function create(): void
    {
        $user = Auth::requireUser();
        if (!Auth::checkCsrf($_POST['csrf'] ?? '')) { http_response_code(400); echo 'Bad CSRF'; return; }
        $name = trim($_POST['name'] ?? '');
        $vis = $_POST['visibility'] ?? 'private';
        if ($name==='') { header('Location: /watchlists'); return; }
        $slug = Utils::slug($name);
        $pdo = Database::pdo();
        $i=1; $base=$slug;
        while ((int)$pdo->prepare('SELECT COUNT(*) FROM watchlists WHERE user_id = :u AND slug = :s')->execute([':u'=>(int)$user['id'], ':s'=>$slug]) && (int)$pdo->query("SELECT FOUND_ROWS()")===1) { $i++; $slug=$base.'-'.$i; }
        $ins = $pdo->prepare('INSERT INTO watchlists (user_id, name, slug, visibility, is_default) VALUES (:u,:n,:s,:v,0)');
        $ins->execute([':u'=>(int)$user['id'], ':n'=>$name, ':s'=>$slug, ':v'=>$vis]);
        header('Location: /watchlists');
    }

    public function showPublic(string $username, string $slug): void
    {
        $pdo = Database::pdo();
        // Find user by username slug (best-effort: match against name or email localpart)
        $userStmt = $pdo->prepare('SELECT id, name, email FROM users');
        $userStmt->execute();
        $owner = null;
        foreach ($userStmt->fetchAll() as $u) { if (Utils::usernameSlug($u) === strtolower($username)) { $owner=$u; break; } }
        if (!$owner) { http_response_code(404); echo View::render('errors/404',['title'=>'User Not Found']); return; }
        $wl = $pdo->prepare('SELECT id, name, slug, visibility FROM watchlists WHERE user_id = :u AND slug = :s LIMIT 1');
        $wl->execute([':u'=>(int)$owner['id'], ':s'=>$slug]);
        $list = $wl->fetch();
        if (!$list) { http_response_code(404); echo View::render('errors/404',['title'=>'Watchlist Not Found']); return; }
        $viewer = Auth::user();
        if ($list['visibility'] !== 'public' && (!$viewer || (int)$viewer['id'] !== (int)$owner['id'])) {
            http_response_code(403); echo View::render('errors/404',['title'=>'Not Accessible']); return;
        }
        $coins = $pdo->prepare('SELECT c.slug, c.name, c.symbol FROM watchlist_items wi JOIN coins c ON c.id = wi.coin_id WHERE wi.watchlist_id = :w');
        $coins->execute([':w'=>(int)$list['id']]);
        echo View::render('watchlists/show', [
            'title'=> $list['name'].' — Watchlist', 'owner'=>$owner, 'list'=>$list, 'coins'=>$coins->fetchAll()
        ]);
    }

    public function exportCsv(string $username, string $slug): void
    {
        $pdo = Database::pdo();
        $userStmt = $pdo->prepare('SELECT id, name, email FROM users');
        $userStmt->execute();
        $owner = null; foreach ($userStmt->fetchAll() as $u) { if (Utils::usernameSlug($u) === strtolower($username)) { $owner=$u; break; } }
        if (!$owner) { http_response_code(404); echo 'Not found'; return; }
        $wl = $pdo->prepare('SELECT id, name, slug, visibility FROM watchlists WHERE user_id = :u AND slug = :s LIMIT 1');
        $wl->execute([':u'=>(int)$owner['id'], ':s'=>$slug]);
        $list = $wl->fetch(); if (!$list) { http_response_code(404); echo 'Not found'; return; }
        $viewer = Auth::user();
        if ($list['visibility'] !== 'public' && (!$viewer || (int)$viewer['id'] !== (int)$owner['id'])) { http_response_code(403); echo 'Forbidden'; return; }
        $coins = $pdo->prepare('SELECT c.slug, c.name, c.symbol FROM watchlist_items wi JOIN coins c ON c.id = wi.coin_id WHERE wi.watchlist_id = :w');
        $coins->execute([':w'=>(int)$list['id']]);
        header('Content-Type: text/csv');
        header('Content-Disposition: attachment; filename="watchlist-'.preg_replace('/[^a-z0-9\-]/i','-', $list['slug']).'.csv"');
        $out = fopen('php://output','w');
        fputcsv($out, ['name','symbol','slug']);
        foreach ($coins->fetchAll() as $r) { fputcsv($out, [$r['name'],$r['symbol'],$r['slug']]); }
        fclose($out);
    }
}
