<?php
namespace App\Models;

use App\Database;
use PDO;

class Exchange
{
    public static function dataTable(int $start, int $length, string $sort, string $dir, ?string $search): array
    {
        $pdo = Database::pdo();
        $sortMap = [
            'name' => 'e.name',
            'trust_score' => 'e.trust_score',
            'volume_24h' => 'v.volume_24h'
        ];
        $sortCol = $sortMap[$sort] ?? 'e.trust_score';
        $dir = strtoupper($dir) === 'DESC' ? 'DESC' : 'ASC';

        // Aggregate 24h volume from markets table (denorm recommended for scale)
        $baseWhere = '1=1';
        $total = (int)$pdo->query('SELECT COUNT(*) FROM exchanges e WHERE ' . $baseWhere)->fetchColumn();

        $params = [];
        $where = $baseWhere;
        if ($search) {
            $where .= ' AND (e.name LIKE :q OR e.country LIKE :q)';
            $params[':q'] = '%' . $search . '%';
        }
        $stmt = $pdo->prepare('SELECT COUNT(*) FROM exchanges e WHERE ' . $where);
        $stmt->execute($params);
        $filtered = (int)$stmt->fetchColumn();

        $sql = "
            SELECT e.name, e.slug, e.country, e.trust_score, e.logo,
                   COALESCE(SUM(em.volume_24h),0) AS volume_24h
            FROM exchanges e
            LEFT JOIN exchange_markets em ON em.exchange_id = e.id
            WHERE $where
            GROUP BY e.id
            ORDER BY $sortCol $dir
            LIMIT :length OFFSET :start
        ";
        $stmt = $pdo->prepare($sql);
        foreach ($params as $k=>$v) $stmt->bindValue($k,$v);
        $stmt->bindValue(':length',$length,PDO::PARAM_INT);
        $stmt->bindValue(':start',$start,PDO::PARAM_INT);
        $stmt->execute();
        $rows = $stmt->fetchAll();
        return [$rows, $total, $filtered];
    }

    public static function findBySlug(string $slug): ?array
    {
        $pdo = Database::pdo();
        $stmt = $pdo->prepare('SELECT * FROM exchanges WHERE slug = :slug LIMIT 1');
        $stmt->execute([':slug'=>$slug]);
        $row = $stmt->fetch();
        return $row ?: null;
    }

    public static function marketsByExchangeId(int $exchangeId, int $start, int $length, string $sort, string $dir, ?string $search): array
    {
        $pdo = Database::pdo();
        $sortMap = [
            'pair' => 'pair',
            'last_price' => 'last_price',
            'volume_24h' => 'volume_24h',
        ];
        $sortCol = $sortMap[$sort] ?? 'volume_24h';
        $dir = strtoupper($dir) === 'DESC' ? 'DESC' : 'ASC';
        $params = [':ex'=>$exchangeId];
        $where = 'exchange_id = :ex';
        if ($search) {
            $where .= ' AND (pair LIKE :q OR base_symbol LIKE :q OR quote_symbol LIKE :q)';
            $params[':q'] = '%' . $search . '%';
        }
        $stmt = $pdo->prepare('SELECT COUNT(*) FROM exchange_markets WHERE ' . $where);
        $stmt->execute($params);
        $filtered = (int)$stmt->fetchColumn();
        $totalStmt = $pdo->prepare('SELECT COUNT(*) FROM exchange_markets WHERE exchange_id = :ex');
        $totalStmt->execute([':ex'=>$exchangeId]);
        $total = (int)$totalStmt->fetchColumn();

        $sql = "SELECT pair, base_symbol, quote_symbol, last_price, volume_24h, is_spot
                FROM exchange_markets
                WHERE $where
                ORDER BY $sortCol $dir
                LIMIT :length OFFSET :start";
        $stmt = $pdo->prepare($sql);
        foreach ($params as $k=>$v) $stmt->bindValue($k,$v);
        $stmt->bindValue(':length',$length,PDO::PARAM_INT);
        $stmt->bindValue(':start',$start,PDO::PARAM_INT);
        $stmt->execute();
        $rows = $stmt->fetchAll();
        return [$rows, $total, $filtered];
    }
}

