<?php
namespace App;

class Router
{
    private array $routes = ['GET' => [], 'POST' => []];
    private $fallback;

    public function get(string $pattern, $handler): void { $this->map('GET', $pattern, $handler); }
    public function post(string $pattern, $handler): void { $this->map('POST', $pattern, $handler); }

    private function map(string $method, string $pattern, $handler): void
    {
        $this->routes[$method][] = [$this->compile($pattern), $handler];
    }

    public function fallback($handler): void
    {
        $this->fallback = $handler;
    }

    private function compile(string $pattern): array
    {
        // Convert `/coins/{slug}` -> regex with named captures
        $regex = preg_replace('#\{([a-zA-Z_][a-zA-Z0-9_]*)\}#', '(?P<$1>[^/]+)', $pattern);
        $regex = '#^' . $regex . '$#';
        return [$pattern, $regex];
    }

    public function dispatch(string $method, string $uri): void
    {
        $path = parse_url($uri, PHP_URL_PATH) ?? '/';
        $routes = $this->routes[$method] ?? [];
        foreach ($routes as [$compiled, $handler]) {
            [, $regex] = $compiled;
            if (preg_match($regex, $path, $m)) {
                $params = array_filter($m, 'is_string', ARRAY_FILTER_USE_KEY);
                return $this->invoke($handler, $params);
            }
        }
        if ($this->fallback) {
            return ($this->fallback)();
        }
        http_response_code(404);
        echo 'Not Found';
    }

    private function invoke($handler, array $params)
    {
        if (is_callable($handler)) {
            return $handler(...array_values($params));
        }
        if (is_array($handler) && count($handler) === 2) {
            [$class, $method] = $handler;
            $obj = new $class();
            return $obj->$method(...array_values($params));
        }
        throw new \RuntimeException('Invalid route handler');
    }
}

