<?php
namespace App;

use App\Database;
use PDO;

class ScreenEngine
{
    private static array $fieldMap = [
        'rank' => 'c.rank',
        'price' => 'ms.price',
        'market_cap' => 'ms.market_cap',
        'mcap' => 'ms.market_cap',
        'volume_24h' => 'ms.volume_24h',
        'vol' => 'ms.volume_24h',
        'change_1h' => 'ms.change_1h',
        'change_24h' => 'ms.change_24h',
        'change_7d' => 'ms.change_7d',
        'change_30d' => 'ms.change_30d',
        'ath_drawdown' => 'ms.ath_drawdown',
        'name' => 'c.name',
        'symbol' => 'c.symbol',
        'category' => 'cat.name',
    ];

    public static function run(string $dsl, string $currency, int $maxLimit = 200): array
    {
        $currency = strtolower($currency);
        $dsl = trim($dsl);
        $orderBy = null; $orderDir = 'DESC'; $limit = null; $whereParts = []; $params = [];

        // Extract ORDER BY and LIMIT (case-insensitive)
        $rest = $dsl;
        if (preg_match('/\bORDER\s+BY\b/i', $rest, $m, PREG_OFFSET_CAPTURE)) {
            $pos = $m[0][1];
            $before = trim(substr($rest, 0, $pos));
            $after = trim(substr($rest, $pos));
            $rest = $before;
            // parse order by and optional limit from $after
            if (preg_match('/^ORDER\s+BY\s+([a-zA-Z0-9_]+)(?:\s+(ASC|DESC))?/i', $after, $om)) {
                $fld = strtolower($om[1]);
                $dir = strtoupper($om[2] ?? 'DESC');
                if (isset(self::$fieldMap[$fld])) {
                    $orderBy = self::$fieldMap[$fld];
                    $orderDir = $dir === 'ASC' ? 'ASC' : 'DESC';
                }
                // find LIMIT after order by
                if (preg_match('/\bLIMIT\s+(\d+)/i', $after, $lm)) {
                    $limit = (int)$lm[1];
                }
            }
        }
        if ($limit === null && preg_match('/\bLIMIT\s+(\d+)/i', $dsl, $lm)) {
            $limit = (int)$lm[1];
            // remove limit from where portion
            $rest = trim(preg_replace('/\bLIMIT\s+\d+/i', '', $rest));
        }
        if ($limit === null) $limit = 50;
        $limit = max(1, min($maxLimit, $limit));

        // Parse expression with OR/AND and parentheses; support numeric comparisons and string equals/LIKE
        $needCategoryJoin = false;
        $paramIndex = 0;
        $tokens = self::tokenize($rest);
        $ast = self::parseExpression($tokens);
        $build = function($node) use (&$build, &$params, &$paramIndex, &$needCategoryJoin): string {
            if (!$node) return '1=1';
            if ($node['type'] === 'binop') {
                $left = $build($node['left']);
                $right = $build($node['right']);
                return '(' . $left . ' ' . $node['op'] . ' ' . $right . ')';
            }
            if ($node['type'] === 'cond') {
                $fld = strtolower($node['field']);
                if (!isset(self::$fieldMap[$fld])) return '1=1';
                if ($fld === 'category') $needCategoryJoin = true;
                $col = self::$fieldMap[$fld];
                $op = strtoupper($node['op']);
                $ph = ':p' . (++$paramIndex);
                if ($op === 'LIKE') {
                    $params[$ph] = '%' . $node['value'] . '%';
                    return "$col LIKE $ph";
                }
                if (is_numeric($node['value'])) {
                    $params[$ph] = (float)$node['value'];
                } else {
                    $params[$ph] = (string)$node['value'];
                }
                return "$col $op $ph";
            }
            return '1=1';
        };
        $whereExpr = $build($ast);
        if (trim($whereExpr) !== '') $whereParts[] = $whereExpr;

        $whereSql = $whereParts ? ('AND ' . implode(' AND ', $whereParts)) : '';
        $orderSql = $orderBy ? ("ORDER BY $orderBy $orderDir") : 'ORDER BY c.rank ASC';

        $joinCat = $needCategoryJoin ? 'LEFT JOIN coin_categories cc ON cc.coin_id = c.id LEFT JOIN categories cat ON cat.id = cc.category_id' : '';
        $sql = "
            SELECT c.rank, c.name, c.slug, c.symbol,
                   ms.price, ms.market_cap, ms.volume_24h, ms.change_24h
            FROM coins c
            $joinCat
            LEFT JOIN coin_market_snapshots ms
              ON ms.id = (
                  SELECT id FROM coin_market_snapshots
                  WHERE coin_id = c.id AND currency = :currency
                  ORDER BY timestamp DESC LIMIT 1
              )
            WHERE c.is_active = 1
            $whereSql
            $orderSql
            LIMIT :limit
        ";
        $cacheKey = 'screen:' . md5(json_encode([$dsl,$currency,$limit]));
        $cached = CacheStore::get($cacheKey);
        if ($cached !== null) {
            $rows = json_decode($cached, true);
            if (is_array($rows)) return $rows;
        }
        $pdo = Database::pdo();
        $stmt = $pdo->prepare($sql);
        $stmt->bindValue(':currency', $currency);
        foreach ($params as $k=>$v) $stmt->bindValue($k, $v);
        $stmt->bindValue(':limit', $limit, PDO::PARAM_INT);
        $stmt->execute();
        $rows = $stmt->fetchAll();
        CacheStore::set($cacheKey, json_encode($rows), 60);
        return $rows;
    }
}

// Parsing helpers
namespace App;

class ScreenEngineParseError extends \Exception {}

class ScreenEngineTokenizer
{
    public static function tokenize(string $s): array
    {
        $tokens = [];
        $i = 0; $n = strlen($s);
        while ($i < $n) {
            if (ctype_space($s[$i])) { $i++; continue; }
            $ch = $s[$i];
            if ($ch === '(' || $ch === ')') { $tokens[] = ['type'=>'paren','val'=>$ch]; $i++; continue; }
            if (preg_match('/\G(AND|OR)\b/i', $s, $m, 0, $i)) { $tokens[]=['type'=>'op','val'=>strtoupper($m[1])]; $i += strlen($m[0]); continue; }
            if (preg_match('/\G([a-zA-Z_][a-zA-Z0-9_]*)/A', $s, $m, 0, $i)) {
                $tokens[]=['type'=>'ident','val'=>$m[1]]; $i += strlen($m[0]); continue; }
            if (preg_match('/\G(<=|>=|!=|=|<|>|LIKE)\b/i', $s, $m, 0, $i)) { $tokens[]=['type'=>'cmp','val'=>strtoupper($m[1])]; $i += strlen($m[0]); continue; }
            if ($ch === '\'' || $ch === '"') {
                $q=$ch; $j=$i+1; $buf='';
                while ($j<$n && $s[$j]!==$q) { $buf.=$s[$j]; $j++; }
                $tokens[]=['type'=>'string','val'=>$buf]; $i=$j+1; continue;
            }
            if (preg_match('/\G[-+]?\d+(?:\.\d+)?(?:e[-+]?\d+)?/i', $s, $m, 0, $i)) { $tokens[]=['type'=>'number','val'=>$m[0]]; $i += strlen($m[0]); continue; }
            // Unknown char
            $i++;
        }
        return $tokens;
    }
}

class ScreenEngine
{
    // redefine to avoid namespace split; real class above
}

namespace App;

class ScreenEngineParser
{
    private array $tokens; private int $pos=0;
    public function __construct(array $tokens){ $this->tokens=$tokens; }
    private function peek(){ return $this->tokens[$this->pos] ?? null; }
    private function eat(){ return $this->tokens[$this->pos++] ?? null; }
    private function accept($type,$val=null){ $t=$this->peek(); if($t && $t['type']===$type && ($val===null || strtoupper($t['val'])===$val)){ $this->eat(); return $t; } return null; }
    public function parse(){ return $this->parseOr(); }
    private function parseOr(){ $node=$this->parseAnd(); while($this->accept('op','OR')){ $right=$this->parseAnd(); $node=['type'=>'binop','op'=>'OR','left'=>$node,'right'=>$right]; } return $node; }
    private function parseAnd(){ $node=$this->parseTerm(); while($this->accept('op','AND')){ $right=$this->parseTerm(); $node=['type'=>'binop','op'=>'AND','left'=>$node,'right'=>$right]; } return $node; }
    private function parseTerm(){ if($this->accept('paren','(')){ $node=$this->parseOr(); $this->accept('paren',')'); return $node; } return $this->parseCond(); }
    private function parseCond(){ $id=$this->accept('ident'); if(!$id) return null; $cmp=$this->accept('cmp'); if(!$cmp) return null; $valTok = $this->accept('number') ?? $this->accept('string'); if(!$valTok) return null; return ['type'=>'cond','field'=>$id['val'],'op'=>$cmp['val'],'value'=>$valTok['val']]; }
}

namespace App;

class ScreenEngine
{
    public static function tokenize(string $s): array { return ScreenEngineTokenizer::tokenize($s); }
    public static function parseExpression(array $tokens){ $p=new ScreenEngineParser($tokens); return $p->parse(); }
}
