<?php
// CoinGecko `/coins/markets` ingestion with backoff, lock, and upserts

require __DIR__ . '/../../app/bootstrap.php';
require_once __DIR__ . '/../lib/http.php';
require_once __DIR__ . '/../lib/lock.php';

use App\Config;
use App\Database;
use App\Utils;
use App\Snapshot;
use Scripts\Lib\Http;
use Scripts\Lib\HttpException;
use Scripts\Lib\Lock;

if (php_sapi_name() !== 'cli') { echo "Run from CLI\n"; exit(1); }
set_time_limit(0);
ini_set('memory_limit', '512M');

$lock = new Lock('markets_pull');
if (!$lock->acquire()) { echo "Another markets_pull is running.\n"; exit(0); }

$pdo = Database::pdo();
$base = rtrim(Config::get('coingecko.base'), '/');
$vs = Config::get('coingecko.vs_currencies', ['usd']);
$limitPerPage = 250; // CoinGecko max
$timeout = (int)Config::get('coingecko.timeout_sec', 20);

// Simple rate limiter (process-local). Replace with DB/Redis backed for multi-worker.
$calls = 0; $windowStart = time(); $rate = (int)Config::get('coingecko.rate_limit_per_min', 30);
$tick = function() use (&$calls, &$windowStart, $rate) {
    $now = time();
    if ($now - $windowStart >= 60) { $windowStart = $now; $calls = 0; }
    if ($calls >= $rate) { sleep(60 - ($now - $windowStart)); $windowStart = time(); $calls = 0; }
    $calls++;
};

$upsertCoin = $pdo->prepare(
    "INSERT INTO coins (cg_id, symbol, name, slug, image_thumb, image_small, is_active, rank)
     VALUES (:cg_id, :symbol, :name, :slug, :image_thumb, :image_small, 1, :rank)
     ON DUPLICATE KEY UPDATE symbol=VALUES(symbol), name=VALUES(name), slug=VALUES(slug), image_thumb=VALUES(image_thumb), image_small=VALUES(image_small), rank=VALUES(rank), updated_at=NOW()"
);

$getCoinId = $pdo->prepare("SELECT id FROM coins WHERE cg_id = :cg_id");

$insertSnap = $pdo->prepare(
    "INSERT IGNORE INTO coin_market_snapshots (coin_id, currency, price, market_cap, volume_24h, change_1h, change_24h, change_7d, change_30d, ath_drawdown, timestamp)
     VALUES (:coin_id, :currency, :price, :market_cap, :volume_24h, :change_1h, :change_24h, :change_7d, :change_30d, :ath_drawdown, :timestamp)"
);

foreach ($vs as $currency) {
    $page = 1;
    while (true) {
        $tick();
        $url = $base . '/coins/markets?vs_currency=' . urlencode($currency) . '&per_page=' . $limitPerPage . '&page=' . $page . '&price_change_percentage=1h,24h,7d,30d&order=market_cap_desc';
        $attempt = 0; $maxAttempts = 5; $backoff = 1;
        $data = null;
        while ($attempt < $maxAttempts) {
            try {
                $data = Http::getJson($url, $timeout);
                break;
            } catch (HttpException $e) {
                $attempt++;
                if ($e->getStatus() === 429 && $e->getRetryAfter()) { sleep($e->getRetryAfter()); }
                if ($attempt >= $maxAttempts) {
                    fwrite(STDERR, "Failed fetching page $page ($currency): " . $e->getMessage() . "\n");
                    $data = [];
                    break;
                }
                sleep($backoff + random_int(0, 1));
                $backoff = min($backoff * 2, 60);
            } catch (\Throwable $e) {
                $attempt++;
                if ($attempt >= $maxAttempts) { fwrite(STDERR, $e->getMessage() . "\n"); $data = []; break; }
                sleep($backoff + random_int(0, 1));
                $backoff = min($backoff * 2, 60);
            }
        }

        if (empty($data)) { if ($page === 1) { echo "No data for $currency\n"; } break; }

        $pdo->beginTransaction();
        try {
            foreach ($data as $row) {
                $cgId = $row['id'] ?? null; if (!$cgId) continue;
                $symbol = strtolower(trim($row['symbol'] ?? ''));
                $name = trim($row['name'] ?? $symbol);
                $slug = Utils::slug($row['id'] ?? ($name ?: $symbol));
                $imageThumb = $row['image'] ?? null;
                $imageSmall = $imageThumb;
                $rank = isset($row['market_cap_rank']) ? (int)$row['market_cap_rank'] : null;

                $upsertCoin->execute([
                    ':cg_id' => $cgId,
                    ':symbol' => $symbol,
                    ':name' => $name,
                    ':slug' => $slug,
                    ':image_thumb' => $imageThumb,
                    ':image_small' => $imageSmall,
                    ':rank' => $rank,
                ]);

                $getCoinId->execute([':cg_id' => $cgId]);
                $coinId = (int)$getCoinId->fetchColumn();
                if (!$coinId) continue;

                $pct = $row['price_change_percentage_24h_in_currency'] ?? $row['price_change_percentage_24h'] ?? null;
                $pct1h = $row['price_change_percentage_1h_in_currency'] ?? null;
                $pct7d = $row['price_change_percentage_7d_in_currency'] ?? null;
                $pct30d = $row['price_change_percentage_30d_in_currency'] ?? null;
                $athDraw = null; // Compute later from metadata/OHLC if needed
                $ts = date('Y-m-d H:i:s');

                $insertSnap->execute([
                    ':coin_id' => $coinId,
                    ':currency' => strtolower($currency),
                    ':price' => $row['current_price'] ?? null,
                    ':market_cap' => $row['market_cap'] ?? null,
                    ':volume_24h' => $row['total_volume'] ?? null,
                    ':change_1h' => $pct1h,
                    ':change_24h' => $pct,
                    ':change_7d' => $pct7d,
                    ':change_30d' => $pct30d,
                    ':ath_drawdown' => $athDraw,
                    ':timestamp' => $ts,
                ]);

                // Invalidate coin page snapshot
                Snapshot::invalidate('/coins/' . $slug);
            }
            $pdo->commit();
        } catch (\Throwable $e) {
            $pdo->rollBack();
            fwrite(STDERR, "DB error page $page ($currency): " . $e->getMessage() . "\n");
        }

        if (count($data) < $limitPerPage) break; // last page reached
        $page++;
        if ($page > 50) break; // safety cap
    }
}

// Invalidate coins list snapshot
\App\Snapshot::invalidate('/coins');

echo "markets_pull finished\n";
