<?php
// Builds sitemap index and coin sitemaps under public/sitemaps/
require __DIR__ . '/../../app/bootstrap.php';

use App\Config;
use App\Database;

if (php_sapi_name() !== 'cli') { echo "Run from CLI\n"; exit(1); }

$pdo = Database::pdo();
$base = rtrim(Config::get('app.base_url', ''), '/');
$outDir = __DIR__ . '/../../public/sitemaps';
if (!is_dir($outDir)) @mkdir($outDir, 0777, true);

// Build coin sitemaps in chunks of 1000
$count = (int)$pdo->query('SELECT COUNT(*) FROM coins WHERE is_active = 1')->fetchColumn();
$perFile = 1000;
$files = [];
for ($offset = 0, $part = 1; $offset < $count; $offset += $perFile, $part++) {
    $stmt = $pdo->prepare("SELECT c.slug, MAX(ms.timestamp) as lastmod
                           FROM coins c
                           LEFT JOIN coin_market_snapshots ms ON ms.coin_id = c.id
                           WHERE c.is_active = 1
                           GROUP BY c.id
                           ORDER BY c.rank IS NULL, c.rank ASC, c.id ASC
                           LIMIT :limit OFFSET :offset");
    $stmt->bindValue(':limit', $perFile, PDO::PARAM_INT);
    $stmt->bindValue(':offset', $offset, PDO::PARAM_INT);
    $stmt->execute();
    $rows = $stmt->fetchAll();
    if (!$rows) break;
    $xml = new DOMDocument('1.0', 'UTF-8');
    $xml->formatOutput = true;
    $urlset = $xml->createElement('urlset');
    $urlset->setAttribute('xmlns', 'http://www.sitemaps.org/schemas/sitemap/0.9');
    $xml->appendChild($urlset);
    foreach ($rows as $r) {
        $url = $xml->createElement('url');
        $loc = $xml->createElement('loc', htmlspecialchars($base . '/coins/' . $r['slug']));
        $url->appendChild($loc);
        if (!empty($r['lastmod'])) {
            $lm = date('c', strtotime($r['lastmod']));
            $url->appendChild($xml->createElement('lastmod', $lm));
        }
        $urlset->appendChild($url);
    }
    $fname = "sitemap-coins-{$part}.xml";
    $xml->save($outDir . '/' . $fname);
    $files[] = $fname;
}

// Build index
$index = new DOMDocument('1.0', 'UTF-8');
$index->formatOutput = true;
$sitemapindex = $index->createElement('sitemapindex');
$sitemapindex->setAttribute('xmlns', 'http://www.sitemaps.org/schemas/sitemap/0.9');
$index->appendChild($sitemapindex);
foreach ($files as $f) {
    $sm = $index->createElement('sitemap');
    $sm->appendChild($index->createElement('loc', htmlspecialchars($base . '/sitemaps/' . $f)));
    $sm->appendChild($index->createElement('lastmod', date('c')));
    $sitemapindex->appendChild($sm);
}

// Categories sitemap
$cats = $pdo->query('SELECT slug, updated_at FROM categories ORDER BY id ASC')->fetchAll();
if ($cats) {
    $xml = new DOMDocument('1.0', 'UTF-8');
    $xml->formatOutput = true;
    $urlset = $xml->createElement('urlset');
    $urlset->setAttribute('xmlns', 'http://www.sitemaps.org/schemas/sitemap/0.9');
    $xml->appendChild($urlset);
    foreach ($cats as $r) {
        $url = $xml->createElement('url');
        $loc = $xml->createElement('loc', htmlspecialchars($base . '/categories/' . $r['slug']));
        $url->appendChild($loc);
        if (!empty($r['updated_at'])) {
            $url->appendChild($xml->createElement('lastmod', date('c', strtotime($r['updated_at']))));
        }
        $urlset->appendChild($url);
    }
    $fname = 'sitemap-categories.xml';
    $xml->save($outDir . '/' . $fname);
    $sm = $index->createElement('sitemap');
    $sm->appendChild($index->createElement('loc', htmlspecialchars($base . '/sitemaps/' . $fname)));
    $sm->appendChild($index->createElement('lastmod', date('c')));
    $sitemapindex->appendChild($sm);
}

$index->save($outDir . '/sitemap-index.xml');

echo "Built " . count($files) . " coin sitemap(s) and index.\n";

// Exchanges sitemap (single file)
$exchanges = $pdo->query('SELECT slug, updated_at FROM exchanges ORDER BY trust_score DESC, id ASC')->fetchAll();
if ($exchanges) {
    $xml = new DOMDocument('1.0', 'UTF-8');
    $xml->formatOutput = true;
    $urlset = $xml->createElement('urlset');
    $urlset->setAttribute('xmlns', 'http://www.sitemaps.org/schemas/sitemap/0.9');
    $xml->appendChild($urlset);
    foreach ($exchanges as $r) {
        $url = $xml->createElement('url');
        $loc = $xml->createElement('loc', htmlspecialchars($base . '/exchanges/' . $r['slug']));
        $url->appendChild($loc);
        if (!empty($r['updated_at'])) {
            $url->appendChild($xml->createElement('lastmod', date('c', strtotime($r['updated_at']))));
        }
        $urlset->appendChild($url);
    }
    $fname = 'sitemap-exchanges.xml';
    $xml->save($outDir . '/' . $fname);
    // Append to index too
    $index = new DOMDocument('1.0', 'UTF-8');
    $index->formatOutput = true;
    $sitemapindex = $index->createElement('sitemapindex');
    $sitemapindex->setAttribute('xmlns', 'http://www.sitemaps.org/schemas/sitemap/0.9');
    $index->appendChild($sitemapindex);
    foreach (array_merge($files, ['sitemap-categories.xml', $fname]) as $f) {
        $sm = $sitemapindex->appendChild($index->createElement('sitemap'));
        $sm->appendChild($index->createElement('loc', htmlspecialchars($base . '/sitemaps/' . $f)));
        $sm->appendChild($index->createElement('lastmod', date('c')));
    }
    $index->save($outDir . '/sitemap-index.xml');
    echo "Added exchanges sitemap and updated index.\n";
}

// Screens sitemap (public only)
$screens = $pdo->query("SELECT slug, updated_at FROM screens WHERE visibility = 'public' ORDER BY id DESC LIMIT 5000")->fetchAll();
if ($screens) {
    $xml = new DOMDocument('1.0', 'UTF-8');
    $xml->formatOutput = true;
    $urlset = $xml->createElement('urlset');
    $urlset->setAttribute('xmlns', 'http://www.sitemaps.org/schemas/sitemap/0.9');
    $xml->appendChild($urlset);
    foreach ($screens as $r) {
        $url = $xml->createElement('url');
        $loc = $xml->createElement('loc', htmlspecialchars($base . '/screens/' . $r['slug']));
        $url->appendChild($loc);
        if (!empty($r['updated_at'])) {
            $url->appendChild($xml->createElement('lastmod', date('c', strtotime($r['updated_at']))));
        }
        $urlset->appendChild($url);
    }
    $fname = 'sitemap-screens.xml';
    $xml->save($outDir . '/' . $fname);
    // update index
    $index = new DOMDocument('1.0', 'UTF-8');
    $index->formatOutput = true;
    $sitemapindex = $index->createElement('sitemapindex');
    $sitemapindex->setAttribute('xmlns', 'http://www.sitemaps.org/schemas/sitemap/0.9');
    $index->appendChild($sitemapindex);
    foreach (array_merge($files, ['sitemap-categories.xml', 'sitemap-exchanges.xml', $fname]) as $f) {
        $sm = $sitemapindex->appendChild($index->createElement('sitemap'));
        $sm->appendChild($index->createElement('loc', htmlspecialchars($base . '/sitemaps/' . $f)));
        $sm->appendChild($index->createElement('lastmod', date('c')));
    }
    $index->save($outDir . '/sitemap-index.xml');
    echo "Added screens sitemap and updated index.\n";
}
